<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\ApprovalAction;
use App\Models\Submission;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class ApprovalActionController extends Controller
{
    /**
     * List approval actions (audit trail)
     * GET /api/approval-actions?submission_id=&action=&acted_by=&per_page=
     */
    public function index(Request $request)
    {
        $q = ApprovalAction::with([
                'actor:id,name,email',
                'submission:id,status,year,month,institution_id'
            ])
            ->orderByDesc('created_at');

        if ($submissionId = $request->query('submission_id')) {
            $q->where('submission_id', $submissionId);
        }
        if ($action = $request->query('action')) {
            $q->where('action', $action);
        }
        if ($actorId = $request->query('acted_by')) {
            $q->where('acted_by', $actorId);
        }

        return response()->json([
            'message' => 'Approval actions retrieved successfully.',
            'data' => $q->paginate($request->query('per_page', 15)),
        ]);
    }

    /**
     * Show one approval action
     * GET /api/approval-actions/{approvalAction}
     */
    public function show(ApprovalAction $approvalAction)
    {
        $approvalAction->load(['actor', 'submission']);

        return response()->json([
            'message' => 'Approval action retrieved successfully.',
            'data' => $approvalAction
        ]);
    }

    // ---------------------------------------------------------------------
    // WORKFLOW OPERATIONS (all in this controller, as requested)
    // ---------------------------------------------------------------------

    /**
     * POST /api/submissions/{submission}/submit
     */
    public function submit(Request $request, Submission $submission)
    {
        return DB::transaction(function () use ($request, $submission) {
            $from = $submission->status;

            if ($from !== 'DRAFT') {
                return response()->json([
                    'message' => 'Only DRAFT submissions can be submitted.',
                    'data' => $submission
                ], 422);
            }

            $submission->update([
                'status' => 'SUBMITTED',
                'rejection_reason' => null,
            ]);

            $this->logAction($request, $submission, 'SUBMIT', 'Submitted for validation.', $from, 'SUBMITTED');

            return response()->json([
                'message' => 'Submission submitted successfully.',
                'data' => $submission->fresh()
            ]);
        });
    }

    /**
     * POST /api/submissions/{submission}/validate
     * Minimal validation example:
     * - must have at least 1 raw record OR at least 1 evidence file
     */
    public function validateSubmission(Request $request, Submission $submission)
    {
        return DB::transaction(function () use ($request, $submission) {
            $from = $submission->status;

            if (!in_array($from, ['SUBMITTED', 'REJECTED'], true)) {
                return response()->json([
                    'message' => 'Submission must be SUBMITTED (or REJECTED) to validate.',
                    'data' => $submission
                ], 422);
            }

            // Minimal checks (customize to your NIIMS rules engine)
            $hasRaw  = method_exists($submission, 'rawRecords') ? $submission->rawRecords()->exists() : false;
            $hasFile = method_exists($submission, 'files') ? $submission->files()->exists() : false;

            if (!$hasRaw && !$hasFile) {
                // Fail validation
                $submission->update([
                    'status' => 'REJECTED',
                    'rejection_reason' => 'Validation failed: submission must include raw records or evidence files.'
                ]);

                $this->logAction(
                    $request,
                    $submission,
                    'REJECT',
                    $submission->rejection_reason,
                    $from,
                    'REJECTED'
                );

                return response()->json([
                    'message' => 'Validation failed. Submission rejected.',
                    'data' => $submission->fresh()
                ], 422);
            }

            // Pass validation
            $submission->update([
                'status' => 'VALIDATED',
                'rejection_reason' => null
            ]);

            $this->logAction($request, $submission, 'VALIDATE', 'Validation passed.', $from, 'VALIDATED');

            return response()->json([
                'message' => 'Submission validated successfully.',
                'data' => $submission->fresh()
            ]);
        });
    }

    /**
     * POST /api/submissions/{submission}/approve
     */
    public function approve(Request $request, Submission $submission)
    {
        return DB::transaction(function () use ($request, $submission) {
            $from = $submission->status;

            if ($from !== 'VALIDATED') {
                return response()->json([
                    'message' => 'Only VALIDATED submissions can be approved.',
                    'data' => $submission
                ], 422);
            }

            $submission->update([
                'status' => 'APPROVED',
                'rejection_reason' => null
            ]);

            $this->logAction($request, $submission, 'APPROVE', 'Approved.', $from, 'APPROVED');

            return response()->json([
                'message' => 'Submission approved successfully.',
                'data' => $submission->fresh()
            ]);
        });
    }

    /**
     * POST /api/submissions/{submission}/publish
     */
    public function publish(Request $request, Submission $submission)
    {
        return DB::transaction(function () use ($request, $submission) {
            $from = $submission->status;

            if ($from !== 'APPROVED') {
                return response()->json([
                    'message' => 'Only APPROVED submissions can be published.',
                    'data' => $submission
                ], 422);
            }

            $submission->update([
                'status' => 'PUBLISHED',
                'rejection_reason' => null
            ]);

            $this->logAction($request, $submission, 'PUBLISH', 'Published for reporting.', $from, 'PUBLISHED');

            return response()->json([
                'message' => 'Submission published successfully.',
                'data' => $submission->fresh()
            ]);
        });
    }

    /**
     * POST /api/submissions/{submission}/reject
     * body: { "reason": "..." }
     */
    public function reject(Request $request, Submission $submission)
    {
        $request->validate([
            'reason' => ['required', 'string', 'max:2000']
        ]);

        return DB::transaction(function () use ($request, $submission) {
            $from = $submission->status;

            $submission->update([
                'status' => 'REJECTED',
                'rejection_reason' => $request->input('reason')
            ]);

            $this->logAction($request, $submission, 'REJECT', $request->input('reason'), $from, 'REJECTED');

            return response()->json([
                'message' => 'Submission rejected successfully.',
                'data' => $submission->fresh()
            ]);
        });
    }

    // ---------------------------------------------------------------------
    // Helper: writes to approval_actions table (single source of workflow truth)
    // ---------------------------------------------------------------------
    private function logAction(Request $request, Submission $submission, string $action, ?string $comment, ?string $from, ?string $to): void
    {
        ApprovalAction::create([
            'submission_id' => $submission->id,
            'acted_by'      => $request->user()?->id,
            'action'        => $action,
            'comment'       => $comment,
            'from_status'   => $from,
            'to_status'     => $to,
            'ip_address'    => $request->ip(),
            'user_agent'    => $request->userAgent(),
        ]);
    }
}
