<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AuditLog;
use App\Models\Export;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

class ExportController extends Controller
{
    /**
     * Helper to write audit logs consistently.
     */
    private function audit(Request $request, string $entityType, int $entityId, string $event, ?array $before = null, ?array $after = null, ?string $comment = null): void
    {
        AuditLog::create([
            'user_id'     => $request->user()?->id,
            'entity_type' => $entityType,
            'entity_id'   => $entityId,
            'event'       => $event,
            'before'      => $before,
            'after'       => $after,
            'ip_address'  => $request->ip(),
            'user_agent'  => $request->userAgent(),
            'comment'     => $comment,
        ]);
    }

    /**
     * GET /api/exports?year=&type=&status=
     */
    public function index(Request $request)
    {
        $q = Export::with(['requester:id,name,email', 'giiVersion'])
            ->orderByDesc('created_at');

        if ($year = $request->query('year')) {
            $q->where('year', (int)$year);
        }
        if ($type = $request->query('type')) {
            $q->where('type', $type);
        }
        if ($status = $request->query('status')) {
            $q->where('status', $status);
        }

        $result = $q->paginate($request->query('per_page', 15));

        // Audit: exports listing viewed (entity_id 0 because it's a listing)
        $this->audit(
            $request,
            'Export',
            0,
            'exports.list',
            null,
            [
                'filters' => $request->only(['year', 'type', 'status', 'per_page']),
                'returned' => $result->count(),
                'total' => $result->total(),
                'page' => $result->currentPage(),
            ],
            'Viewed exports list'
        );

        return response()->json([
            'message' => 'Exports retrieved successfully.',
            'data' => $result,
        ]);
    }

    /**
     * GET /api/exports/{export}
     */
    public function show(Request $request, Export $export)
    {
        $export->load(['requester', 'giiVersion']);

        // Audit: export viewed
        $this->audit(
            $request,
            'Export',
            $export->id,
            'exports.view',
            null,
            $export->toArray(),
            'Viewed export'
        );

        return response()->json([
            'message' => 'Export retrieved successfully.',
            'data' => $export,
        ]);
    }

    /**
     * POST /api/exports
     * body: { gii_version_id, year, type }
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'gii_version_id' => ['required', 'exists:gii_versions,id'],
            'year'           => ['required', 'integer', 'min:1900', 'max:2100'],
            'type'           => ['nullable', 'in:wipo,internal,dashboard,api'],
        ]);

        $export = Export::create([
            'requested_by'   => $request->user()?->id,
            'gii_version_id' => $data['gii_version_id'],
            'year'           => $data['year'],
            'type'           => $data['type'] ?? 'wipo',
            'status'         => 'PENDING',
            'meta'           => [
                'request_payload' => $data,
            ],
        ]);

        // Audit: export created
        $this->audit(
            $request,
            'Export',
            $export->id,
            'exports.create',
            null,
            $export->toArray(),
            'Created export request'
        );

        return response()->json([
            'message' => 'Export request created.',
            'data' => $export,
        ], 201);
    }

    /**
     * POST /api/exports/{export}/generate
     * (synchronous example – move to Job for production)
     */
    public function generate(Request $request, Export $export)
    {
        if ($export->status !== 'PENDING') {
            // Audit: generate attempted but blocked
            $this->audit(
                $request,
                'Export',
                $export->id,
                'exports.generate.blocked',
                $export->toArray(),
                $export->toArray(),
                'Generate blocked: export already processed'
            );

            return response()->json([
                'message' => 'Export already processed.',
                'data' => $export
            ], 422);
        }

        $before = $export->toArray();

        try {
            // Example: WIPO-style CSV from published observations
            $rows = DB::table('indicator_observations as io')
                ->join('gii_indicators as gi', 'gi.id', '=', 'io.gii_indicator_id')
                ->where('io.year', $export->year)
                ->where('io.status', 'PUBLISHED')
                ->select('gi.code', 'gi.name', 'io.value', 'io.unit')
                ->orderBy('gi.code')
                ->get();

            $csv = "indicator_code,indicator_name,value,unit\n";
            foreach ($rows as $r) {
                $csv .= sprintf(
                    "\"%s\",\"%s\",%s,\"%s\"\n",
                    $r->code,
                    str_replace('"','""',$r->name),
                    $r->value ?? '',
                    $r->unit ?? ''
                );
            }

            $path = "niims/exports/export_{$export->id}.csv";
            Storage::disk('public')->put($path, $csv);

            $export->update([
                'status' => 'GENERATED',
                'path'   => $path,
                'error'  => null,
                'meta'   => array_merge((array)($export->meta ?? []), [
                    'generated_at' => now()->toISOString(),
                    'rows' => $rows->count(),
                ]),
            ]);

            // Audit: generation success
            $this->audit(
                $request,
                'Export',
                $export->id,
                'exports.generate.success',
                $before,
                $export->fresh()->toArray(),
                "Generated export file with {$rows->count()} rows"
            );
        } catch (\Throwable $e) {
            $export->update([
                'status' => 'FAILED',
                'error'  => $e->getMessage(),
                'meta'   => array_merge((array)($export->meta ?? []), [
                    'failed_at' => now()->toISOString(),
                ]),
            ]);

            // Audit: generation failed
            $this->audit(
                $request,
                'Export',
                $export->id,
                'exports.generate.failed',
                $before,
                $export->fresh()->toArray(),
                'Export generation failed: ' . $e->getMessage()
            );
        }

        return response()->json([
            'message' => 'Export generation completed.',
            'data' => $export->fresh(),
        ]);
    }

    /**
     * GET /api/exports/{export}/download
     */
    public function download(Request $request, Export $export)
    {
        $before = $export->toArray();

        if (!$export->path || !Storage::disk('public')->exists($export->path)) {
            // Audit: download failed (not found)
            $this->audit(
                $request,
                'Export',
                $export->id,
                'exports.download.not_found',
                $before,
                $before,
                'Download failed: file not found'
            );

            return response()->json([
                'message' => 'Export file not found.'
            ], 404);
        }

        // Audit: download success (log metadata, not file contents)
        $this->audit(
            $request,
            'Export',
            $export->id,
            'exports.download',
            $before,
            [
                'path' => $export->path,
                'size' => Storage::disk('public')->size($export->path),
            ],
            'Downloaded export file'
        );

        return response()->download(
            Storage::disk('public')->path($export->path)
        );
    }
}
