<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AuditLog;
use App\Models\GiiIndicator;
use Illuminate\Http\Request;

class GiiIndicatorController extends Controller
{
    private function audit(
        Request $request,
        int $entityId,
        string $event,
        ?array $before = null,
        ?array $after = null,
        ?string $comment = null
    ): void {
        AuditLog::create([
            'user_id'     => $request->user()?->id,
            'entity_type' => 'GiiIndicator',
            'entity_id'   => $entityId,
            'event'       => $event,
            'before'      => $before,
            'after'       => $after,
            'ip_address'  => $request->ip(),
            'user_agent'  => $request->userAgent(),
            'comment'     => $comment,
        ]);
    }

    /**
     * GET /api/gii-indicators?gii_subpillar_id=&code=&periodicity=&is_active=&per_page=
     */
    public function index(Request $request)
    {
        $q = GiiIndicator::with(['subpillar:id,gii_pillar_id,code,name'])
            ->orderBy('display_order')
            ->orderBy('code');

        if ($subpillarId = $request->query('gii_subpillar_id')) {
            $q->where('gii_subpillar_id', (int)$subpillarId);
        }
        if ($code = $request->query('code')) {
            $q->where('code', 'like', "%{$code}%");
        }
        if ($periodicity = $request->query('periodicity')) {
            $q->where('periodicity', $periodicity);
        }
        if (!is_null($request->query('is_active'))) {
            $q->where('is_active', (bool)$request->query('is_active'));
        }

        $result = $q->paginate($request->query('per_page', 15));

        $this->audit(
            $request,
            0,
            'gii_indicators.list',
            null,
            [
                'filters' => $request->only(['gii_subpillar_id', 'code', 'periodicity', 'is_active', 'per_page']),
                'returned' => $result->count(),
                'total' => $result->total(),
                'page' => $result->currentPage(),
            ],
            'Viewed GII indicators list'
        );

        return response()->json([
            'message' => 'GII indicators retrieved successfully.',
            'data' => $result,
        ]);
    }

    /**
     * GET /api/gii-indicators/{giiIndicator}
     */
    public function show(Request $request, GiiIndicator $giiIndicator)
    {
        $giiIndicator->load(['subpillar', 'formula']);

        $this->audit(
            $request,
            $giiIndicator->id,
            'gii_indicators.view',
            null,
            $giiIndicator->toArray(),
            'Viewed GII indicator'
        );

        return response()->json([
            'message' => 'GII indicator retrieved successfully.',
            'data' => $giiIndicator,
        ]);
    }

    /**
     * POST /api/gii-indicators
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'gii_subpillar_id' => ['required', 'exists:gii_subpillars,id'],
            'code'             => ['required', 'string', 'max:50'],
            'name'             => ['required', 'string', 'max:255'],
            'unit'             => ['nullable', 'string', 'max:50'],
            'periodicity'      => ['nullable', 'in:yearly,quarterly,monthly'],
            'is_computed'      => ['nullable', 'boolean'],
            'display_order'    => ['nullable', 'integer', 'min:0'],
            'meta'             => ['nullable', 'array'],
        ]);

        $indicator = GiiIndicator::create([
            'gii_subpillar_id' => $data['gii_subpillar_id'],
            'code'             => $data['code'],
            'name'             => $data['name'],
            'unit'             => $data['unit'] ?? null,
            'periodicity'      => $data['periodicity'] ?? 'yearly',
            'is_computed'      => (bool)($data['is_computed'] ?? false),
            'display_order'    => $data['display_order'] ?? 0,
            'meta'             => $data['meta'] ?? null,
            'is_active'        => true,
        ]);

        $this->audit(
            $request,
            $indicator->id,
            'gii_indicators.create',
            null,
            $indicator->toArray(),
            'Created GII indicator'
        );

        return response()->json([
            'message' => 'GII indicator created successfully.',
            'data' => $indicator,
        ], 201);
    }

    /**
     * PUT /api/gii-indicators/{giiIndicator}
     */
    public function update(Request $request, GiiIndicator $giiIndicator)
    {
        $before = $giiIndicator->toArray();

        $data = $request->validate([
            'code'          => ['sometimes', 'string', 'max:50'],
            'name'          => ['sometimes', 'string', 'max:255'],
            'unit'          => ['nullable', 'string', 'max:50'],
            'periodicity'   => ['sometimes', 'in:yearly,quarterly,monthly'],
            'is_computed'   => ['sometimes', 'boolean'],
            'is_active'     => ['sometimes', 'boolean'],
            'display_order' => ['sometimes', 'integer', 'min:0'],
            'meta'          => ['nullable', 'array'],
        ]);

        $giiIndicator->update($data);

        $this->audit(
            $request,
            $giiIndicator->id,
            'gii_indicators.update',
            $before,
            $giiIndicator->fresh()->toArray(),
            'Updated GII indicator'
        );

        return response()->json([
            'message' => 'GII indicator updated successfully.',
            'data' => $giiIndicator->fresh(),
        ]);
    }

    /**
     * PATCH /api/gii-indicators/{giiIndicator}/toggle
     */
    public function toggle(Request $request, GiiIndicator $giiIndicator)
    {
        $before = $giiIndicator->toArray();

        $giiIndicator->update([
            'is_active' => !$giiIndicator->is_active,
        ]);

        $this->audit(
            $request,
            $giiIndicator->id,
            'gii_indicators.toggle',
            $before,
            $giiIndicator->fresh()->toArray(),
            'Toggled indicator active state'
        );

        return response()->json([
            'message' => 'GII indicator status updated.',
            'data' => $giiIndicator->fresh(),
        ]);
    }
}
