<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AuditLog;
use App\Models\GiiPillar;
use Illuminate\Http\Request;

class GiiPillarController extends Controller
{
    private function audit(
        Request $request,
        int $entityId,
        string $event,
        ?array $before = null,
        ?array $after = null,
        ?string $comment = null
    ): void {
        AuditLog::create([
            'user_id'     => $request->user()?->id,
            'entity_type' => 'GiiPillar',
            'entity_id'   => $entityId,
            'event'       => $event,
            'before'      => $before,
            'after'       => $after,
            'ip_address'  => $request->ip(),
            'user_agent'  => $request->userAgent(),
            'comment'     => $comment,
        ]);
    }

    /**
     * GET /api/gii-pillars?gii_version_id=&is_active=&per_page=
     */
    public function index(Request $request)
    {
        $q = GiiPillar::with('version:id,year,name,is_active')
            ->orderBy('display_order')
            ->orderBy('pillar_number');

        if ($versionId = $request->query('gii_version_id')) {
            $q->where('gii_version_id', (int)$versionId);
        }
        if (!is_null($request->query('is_active'))) {
            $q->where('is_active', (bool)$request->query('is_active'));
        }

        $result = $q->paginate($request->query('per_page', 15));

        $this->audit(
            $request,
            0,
            'gii_pillars.list',
            null,
            [
                'filters' => $request->only(['gii_version_id', 'is_active', 'per_page']),
                'returned' => $result->count(),
                'total' => $result->total(),
                'page' => $result->currentPage(),
            ],
            'Viewed GII pillars list'
        );

        return response()->json([
            'message' => 'GII pillars retrieved successfully.',
            'data' => $result,
        ]);
    }

    /**
     * GET /api/gii-pillars/{giiPillar}
     */
    public function show(Request $request, GiiPillar $giiPillar)
    {
        $giiPillar->load(['version', 'subpillars']);

        $this->audit(
            $request,
            $giiPillar->id,
            'gii_pillars.view',
            null,
            $giiPillar->toArray(),
            'Viewed GII pillar'
        );

        return response()->json([
            'message' => 'GII pillar retrieved successfully.',
            'data' => $giiPillar,
        ]);
    }

    /**
     * POST /api/gii-pillars
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'gii_version_id' => ['required', 'exists:gii_versions,id'],
            'pillar_number'  => ['required', 'integer', 'between:1,7'],
            'code'           => ['nullable', 'string', 'max:10'],
            'name'           => ['required', 'string', 'max:255'],
            'description'    => ['nullable', 'string'],
            'display_order'  => ['nullable', 'integer', 'min:0'],
        ]);

        $pillar = GiiPillar::create([
            'gii_version_id' => $data['gii_version_id'],
            'pillar_number'  => $data['pillar_number'],
            'code'           => $data['code'] ?? null,
            'name'           => $data['name'],
            'description'    => $data['description'] ?? null,
            'display_order'  => $data['display_order'] ?? 0,
            'is_active'      => true,
        ]);

        $this->audit(
            $request,
            $pillar->id,
            'gii_pillars.create',
            null,
            $pillar->toArray(),
            'Created GII pillar'
        );

        return response()->json([
            'message' => 'GII pillar created successfully.',
            'data' => $pillar,
        ], 201);
    }

    /**
     * PUT /api/gii-pillars/{giiPillar}
     */
    public function update(Request $request, GiiPillar $giiPillar)
    {
        $before = $giiPillar->toArray();

        $data = $request->validate([
            'pillar_number' => ['sometimes', 'integer', 'between:1,7'],
            'code'          => ['nullable', 'string', 'max:10'],
            'name'          => ['sometimes', 'string', 'max:255'],
            'description'   => ['nullable', 'string'],
            'display_order' => ['sometimes', 'integer', 'min:0'],
            'is_active'     => ['sometimes', 'boolean'],
        ]);

        $giiPillar->update($data);

        $this->audit(
            $request,
            $giiPillar->id,
            'gii_pillars.update',
            $before,
            $giiPillar->fresh()->toArray(),
            'Updated GII pillar'
        );

        return response()->json([
            'message' => 'GII pillar updated successfully.',
            'data' => $giiPillar->fresh(),
        ]);
    }

    /**
     * PATCH /api/gii-pillars/{giiPillar}/toggle
     */
    public function toggle(Request $request, GiiPillar $giiPillar)
    {
        $before = $giiPillar->toArray();

        $giiPillar->update([
            'is_active' => !$giiPillar->is_active
        ]);

        $this->audit(
            $request,
            $giiPillar->id,
            'gii_pillars.toggle',
            $before,
            $giiPillar->fresh()->toArray(),
            'Toggled pillar active state'
        );

        return response()->json([
            'message' => 'GII pillar status updated.',
            'data' => $giiPillar->fresh(),
        ]);
    }
}
