<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AuditLog;
use App\Models\GiiSubpillar;
use Illuminate\Http\Request;

class GiiSubpillarController extends Controller
{
    private function audit(
        Request $request,
        int $entityId,
        string $event,
        ?array $before = null,
        ?array $after = null,
        ?string $comment = null
    ): void {
        AuditLog::create([
            'user_id'     => $request->user()?->id,
            'entity_type' => 'GiiSubpillar',
            'entity_id'   => $entityId,
            'event'       => $event,
            'before'      => $before,
            'after'       => $after,
            'ip_address'  => $request->ip(),
            'user_agent'  => $request->userAgent(),
            'comment'     => $comment,
        ]);
    }

    /**
     * GET /api/gii-subpillars?gii_pillar_id=&is_active=&per_page=
     */
    public function index(Request $request)
    {
        $q = GiiSubpillar::with('pillar:id,gii_version_id,pillar_number,name')
            ->orderBy('display_order')
            ->orderBy('name');

        if ($pillarId = $request->query('gii_pillar_id')) {
            $q->where('gii_pillar_id', (int)$pillarId);
        }
        if (!is_null($request->query('is_active'))) {
            $q->where('is_active', (bool)$request->query('is_active'));
        }

        $result = $q->paginate($request->query('per_page', 15));

        $this->audit(
            $request,
            0,
            'gii_subpillars.list',
            null,
            [
                'filters' => $request->only(['gii_pillar_id', 'is_active', 'per_page']),
                'returned' => $result->count(),
                'total' => $result->total(),
                'page' => $result->currentPage(),
            ],
            'Viewed GII subpillars list'
        );

        return response()->json([
            'message' => 'GII subpillars retrieved successfully.',
            'data' => $result,
        ]);
    }

    /**
     * GET /api/gii-subpillars/{giiSubpillar}
     */
    public function show(Request $request, GiiSubpillar $giiSubpillar)
    {
        $giiSubpillar->load(['pillar', 'indicators']);

        $this->audit(
            $request,
            $giiSubpillar->id,
            'gii_subpillars.view',
            null,
            $giiSubpillar->toArray(),
            'Viewed GII subpillar'
        );

        return response()->json([
            'message' => 'GII subpillar retrieved successfully.',
            'data' => $giiSubpillar,
        ]);
    }

    /**
     * POST /api/gii-subpillars
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'gii_pillar_id' => ['required', 'exists:gii_pillars,id'],
            'code'          => ['nullable', 'string', 'max:20'],
            'name'          => ['required', 'string', 'max:255'],
            'description'   => ['nullable', 'string'],
            'display_order' => ['nullable', 'integer', 'min:0'],
        ]);

        $subpillar = GiiSubpillar::create([
            'gii_pillar_id' => $data['gii_pillar_id'],
            'code'          => $data['code'] ?? null,
            'name'          => $data['name'],
            'description'   => $data['description'] ?? null,
            'display_order' => $data['display_order'] ?? 0,
            'is_active'     => true,
        ]);

        $this->audit(
            $request,
            $subpillar->id,
            'gii_subpillars.create',
            null,
            $subpillar->toArray(),
            'Created GII subpillar'
        );

        return response()->json([
            'message' => 'GII subpillar created successfully.',
            'data' => $subpillar,
        ], 201);
    }

    /**
     * PUT /api/gii-subpillars/{giiSubpillar}
     */
    public function update(Request $request, GiiSubpillar $giiSubpillar)
    {
        $before = $giiSubpillar->toArray();

        $data = $request->validate([
            'code'          => ['nullable', 'string', 'max:20'],
            'name'          => ['sometimes', 'string', 'max:255'],
            'description'   => ['nullable', 'string'],
            'display_order' => ['sometimes', 'integer', 'min:0'],
            'is_active'     => ['sometimes', 'boolean'],
        ]);

        $giiSubpillar->update($data);

        $this->audit(
            $request,
            $giiSubpillar->id,
            'gii_subpillars.update',
            $before,
            $giiSubpillar->fresh()->toArray(),
            'Updated GII subpillar'
        );

        return response()->json([
            'message' => 'GII subpillar updated successfully.',
            'data' => $giiSubpillar->fresh(),
        ]);
    }

    /**
     * PATCH /api/gii-subpillars/{giiSubpillar}/toggle
     */
    public function toggle(Request $request, GiiSubpillar $giiSubpillar)
    {
        $before = $giiSubpillar->toArray();

        $giiSubpillar->update([
            'is_active' => !$giiSubpillar->is_active,
        ]);

        $this->audit(
            $request,
            $giiSubpillar->id,
            'gii_subpillars.toggle',
            $before,
            $giiSubpillar->fresh()->toArray(),
            'Toggled subpillar active state'
        );

        return response()->json([
            'message' => 'GII subpillar status updated.',
            'data' => $giiSubpillar->fresh(),
        ]);
    }
}
