<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AuditLog;
use App\Models\GiiVersion;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class GiiVersionController extends Controller
{
    /* --------------------
     | Helper: audit logger
     |--------------------*/
    private function audit(
        Request $request,
        int $entityId,
        string $event,
        ?array $before = null,
        ?array $after = null,
        ?string $comment = null
    ): void {
        AuditLog::create([
            'user_id'     => $request->user()?->id,
            'entity_type' => 'GiiVersion',
            'entity_id'   => $entityId,
            'event'       => $event,
            'before'      => $before,
            'after'       => $after,
            'ip_address'  => $request->ip(),
            'user_agent'  => $request->userAgent(),
            'comment'     => $comment,
        ]);
    }

    /**
     * GET /api/gii-versions
     */
    public function index(Request $request)
    {
        $versions = GiiVersion::orderByDesc('year')->paginate(
            $request->query('per_page', 15)
        );

        $this->audit(
            $request,
            0,
            'gii_versions.list',
            null,
            [
                'count' => $versions->count(),
                'total' => $versions->total(),
            ],
            'Viewed GII versions list'
        );

        return response()->json([
            'message' => 'GII versions retrieved successfully.',
            'data' => $versions,
        ]);
    }

    /**
     * GET /api/gii-versions/{giiVersion}
     */
    public function show(Request $request, GiiVersion $giiVersion)
    {
        $this->audit(
            $request,
            $giiVersion->id,
            'gii_versions.view',
            null,
            $giiVersion->toArray(),
            'Viewed GII version'
        );

        return response()->json([
            'message' => 'GII version retrieved successfully.',
            'data' => $giiVersion,
        ]);
    }

    /**
     * POST /api/gii-versions
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'year' => ['required', 'string', 'max:10', 'unique:gii_versions,year'],
            'name' => ['nullable', 'string', 'max:255'],
        ]);

        $version = GiiVersion::create([
            'year'      => $data['year'],
            'name'      => $data['name'] ?? null,
            'is_active' => false,
        ]);

        $this->audit(
            $request,
            $version->id,
            'gii_versions.create',
            null,
            $version->toArray(),
            'Created GII version'
        );

        return response()->json([
            'message' => 'GII version created successfully.',
            'data' => $version,
        ], 201);
    }

    /**
     * PUT /api/gii-versions/{giiVersion}
     */
    public function update(Request $request, GiiVersion $giiVersion)
    {
        $before = $giiVersion->toArray();

        $data = $request->validate([
            'year' => ['sometimes', 'string', 'max:10', 'unique:gii_versions,year,' . $giiVersion->id],
            'name' => ['nullable', 'string', 'max:255'],
        ]);

        $giiVersion->update($data);

        $this->audit(
            $request,
            $giiVersion->id,
            'gii_versions.update',
            $before,
            $giiVersion->fresh()->toArray(),
            'Updated GII version'
        );

        return response()->json([
            'message' => 'GII version updated successfully.',
            'data' => $giiVersion->fresh(),
        ]);
    }

    /**
     * PATCH /api/gii-versions/{giiVersion}/activate
     * Ensures only ONE version is active
     */
    public function activate(Request $request, GiiVersion $giiVersion)
    {
        return DB::transaction(function () use ($request, $giiVersion) {
            $before = $giiVersion->toArray();

            // Deactivate all others
            GiiVersion::where('is_active', true)->update(['is_active' => false]);

            // Activate selected
            $giiVersion->update(['is_active' => true]);

            $this->audit(
                $request,
                $giiVersion->id,
                'gii_versions.activate',
                $before,
                $giiVersion->fresh()->toArray(),
                'Activated GII version'
            );

            return response()->json([
                'message' => 'GII version activated successfully.',
                'data' => $giiVersion->fresh(),
            ]);
        });
    }
}
