<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AuditLog;
use App\Models\IndicatorFormula;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class IndicatorFormulaController extends Controller
{
    private function audit(
        Request $request,
        int $entityId,
        string $event,
        ?array $before = null,
        ?array $after = null,
        ?string $comment = null
    ): void {
        AuditLog::create([
            'user_id'     => $request->user()?->id,
            'entity_type' => 'IndicatorFormula',
            'entity_id'   => $entityId,
            'event'       => $event,
            'before'      => $before,
            'after'       => $after,
            'ip_address'  => $request->ip(),
            'user_agent'  => $request->userAgent(),
            'comment'     => $comment,
        ]);
    }

    /**
     * GET /api/indicator-formulas?gii_indicator_id=&is_active=&per_page=
     */
    public function index(Request $request)
    {
        $q = IndicatorFormula::with('indicator:id,code,name,gii_subpillar_id')
            ->orderByDesc('created_at');

        if ($indicatorId = $request->query('gii_indicator_id')) {
            $q->where('gii_indicator_id', (int)$indicatorId);
        }
        if (!is_null($request->query('is_active'))) {
            $q->where('is_active', (bool)$request->query('is_active'));
        }

        $result = $q->paginate($request->query('per_page', 15));

        $this->audit(
            $request,
            0,
            'indicator_formulas.list',
            null,
            [
                'filters' => $request->only(['gii_indicator_id', 'is_active', 'per_page']),
                'returned' => $result->count(),
                'total' => $result->total(),
                'page' => $result->currentPage(),
            ],
            'Viewed indicator formulas list'
        );

        return response()->json([
            'message' => 'Indicator formulas retrieved successfully.',
            'data' => $result,
        ]);
    }

    /**
     * GET /api/indicator-formulas/{indicatorFormula}
     */
    public function show(Request $request, IndicatorFormula $indicatorFormula)
    {
        $indicatorFormula->load('indicator');

        $this->audit(
            $request,
            $indicatorFormula->id,
            'indicator_formulas.view',
            null,
            $indicatorFormula->toArray(),
            'Viewed indicator formula'
        );

        return response()->json([
            'message' => 'Indicator formula retrieved successfully.',
            'data' => $indicatorFormula,
        ]);
    }

    /**
     * POST /api/indicator-formulas
     * Creates a new formula version. By default we deactivate existing active formula for that indicator.
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'gii_indicator_id' => ['required', 'exists:gii_indicators,id'],
            'formula_type'     => ['required', 'in:none,expression,sql,php,python'],
            'formula'          => ['nullable', 'string'],
            'params'           => ['nullable', 'array'],
            'checksum'         => ['nullable', 'string', 'size:64'],
        ]);

        return DB::transaction(function () use ($request, $data) {
            // Determine next version number
            $lastVersion = IndicatorFormula::where('gii_indicator_id', $data['gii_indicator_id'])
                ->max('version');

            $newVersion = ($lastVersion ?? 0) + 1;

            // Deactivate existing active formula (recommended governance rule)
            IndicatorFormula::where('gii_indicator_id', $data['gii_indicator_id'])
                ->where('is_active', true)
                ->update(['is_active' => false]);

            $formula = IndicatorFormula::create([
                'gii_indicator_id' => $data['gii_indicator_id'],
                'formula_type'     => $data['formula_type'],
                'formula'          => $data['formula'] ?? null,
                'params'           => $data['params'] ?? null,
                'version'          => $newVersion,
                'is_active'        => true,
                'checksum'         => $data['checksum'] ?? null,
            ]);

            $this->audit(
                $request,
                $formula->id,
                'indicator_formulas.create',
                null,
                $formula->toArray(),
                "Created new formula version {$newVersion} and set active"
            );

            return response()->json([
                'message' => 'Indicator formula created successfully.',
                'data' => $formula,
            ], 201);
        });
    }

    /**
     * PUT /api/indicator-formulas/{indicatorFormula}
     * Updates an existing formula record (not recommended for strict audit; but allowed).
     */
    public function update(Request $request, IndicatorFormula $indicatorFormula)
    {
        $before = $indicatorFormula->toArray();

        $data = $request->validate([
            'formula_type' => ['sometimes', 'in:none,expression,sql,php,python'],
            'formula'      => ['nullable', 'string'],
            'params'       => ['nullable', 'array'],
            'checksum'     => ['nullable', 'string', 'size:64'],
            'is_active'    => ['sometimes', 'boolean'],
        ]);

        // If switching active, ensure only one active per indicator
        if (array_key_exists('is_active', $data) && $data['is_active'] === true) {
            IndicatorFormula::where('gii_indicator_id', $indicatorFormula->gii_indicator_id)
                ->where('id', '!=', $indicatorFormula->id)
                ->update(['is_active' => false]);
        }

        $indicatorFormula->update($data);

        $this->audit(
            $request,
            $indicatorFormula->id,
            'indicator_formulas.update',
            $before,
            $indicatorFormula->fresh()->toArray(),
            'Updated indicator formula'
        );

        return response()->json([
            'message' => 'Indicator formula updated successfully.',
            'data' => $indicatorFormula->fresh(),
        ]);
    }

    /**
     * PATCH /api/indicator-formulas/{indicatorFormula}/activate
     * Activates a chosen formula and deactivates others for the same indicator.
     */
    public function activate(Request $request, IndicatorFormula $indicatorFormula)
    {
        return DB::transaction(function () use ($request, $indicatorFormula) {
            $before = $indicatorFormula->toArray();

            IndicatorFormula::where('gii_indicator_id', $indicatorFormula->gii_indicator_id)
                ->update(['is_active' => false]);

            $indicatorFormula->update(['is_active' => true]);

            $this->audit(
                $request,
                $indicatorFormula->id,
                'indicator_formulas.activate',
                $before,
                $indicatorFormula->fresh()->toArray(),
                'Activated indicator formula'
            );

            return response()->json([
                'message' => 'Indicator formula activated successfully.',
                'data' => $indicatorFormula->fresh(),
            ]);
        });
    }
}
