<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AuditLog;
use App\Models\Institution;
use Illuminate\Http\Request;

class InstitutionController extends Controller
{
    /* --------------------
     | Helper: audit logger
     |--------------------*/
    private function audit(
        Request $request,
        int $entityId,
        string $event,
        ?array $before = null,
        ?array $after = null,
        ?string $comment = null
    ): void {
        AuditLog::create([
            'user_id'     => $request->user()?->id,
            'entity_type' => 'Institution',
            'entity_id'   => $entityId,
            'event'       => $event,
            'before'      => $before,
            'after'       => $after,
            'ip_address'  => $request->ip(),
            'user_agent'  => $request->userAgent(),
            'comment'     => $comment,
        ]);
    }

    /**
     * GET /api/institutions
     */
    public function index(Request $request)
    {
        $q = Institution::query()->orderBy('name');

        if ($type = $request->query('type')) {
            $q->where('type', $type);
        }
        if (!is_null($request->query('is_active'))) {
            $q->where('is_active', (bool)$request->query('is_active'));
        }

        $result = $q->paginate($request->query('per_page', 15));

        $this->audit(
            $request,
            0,
            'institutions.list',
            null,
            [
                'filters' => $request->only(['type', 'is_active', 'per_page']),
                'count'   => $result->count(),
                'total'   => $result->total(),
            ],
            'Viewed institution list'
        );

        return response()->json([
            'message' => 'Institutions retrieved successfully.',
            'data' => $result,
        ]);
    }

    /**
     * GET /api/institutions/{institution}
     */
    public function show(Request $request, Institution $institution)
    {
        $this->audit(
            $request,
            $institution->id,
            'institutions.view',
            null,
            $institution->toArray(),
            'Viewed institution'
        );

        return response()->json([
            'message' => 'Institution retrieved successfully.',
            'data' => $institution,
        ]);
    }

    /**
     * POST /api/institutions
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'name'    => ['required', 'string', 'max:255', 'unique:institutions,name'],
            'type'    => ['required', 'string', 'max:100'],
            'code'    => ['nullable', 'string', 'max:50', 'unique:institutions,code'],
            'email'   => ['nullable', 'email'],
            'phone'   => ['nullable', 'string', 'max:50'],
            'address' => ['nullable', 'string'],
        ]);

        $institution = Institution::create($data);

        $this->audit(
            $request,
            $institution->id,
            'institutions.create',
            null,
            $institution->toArray(),
            'Created institution'
        );

        return response()->json([
            'message' => 'Institution created successfully.',
            'data' => $institution,
        ], 201);
    }

    /**
     * PUT /api/institutions/{institution}
     */
    public function update(Request $request, Institution $institution)
    {
        $before = $institution->toArray();

        $data = $request->validate([
            'name'    => ['sometimes', 'string', 'max:255', 'unique:institutions,name,' . $institution->id],
            'type'    => ['sometimes', 'string', 'max:100'],
            'code'    => ['nullable', 'string', 'max:50', 'unique:institutions,code,' . $institution->id],
            'email'   => ['nullable', 'email'],
            'phone'   => ['nullable', 'string', 'max:50'],
            'address' => ['nullable', 'string'],
            'is_active' => ['sometimes', 'boolean'],
        ]);

        $institution->update($data);

        $this->audit(
            $request,
            $institution->id,
            'institutions.update',
            $before,
            $institution->fresh()->toArray(),
            'Updated institution'
        );

        return response()->json([
            'message' => 'Institution updated successfully.',
            'data' => $institution->fresh(),
        ]);
    }

    /**
     * PATCH /api/institutions/{institution}/toggle
     */
    public function toggle(Request $request, Institution $institution)
    {
        $before = $institution->toArray();

        $institution->update([
            'is_active' => !$institution->is_active,
        ]);

        $this->audit(
            $request,
            $institution->id,
            'institutions.toggle',
            $before,
            $institution->fresh()->toArray(),
            'Toggled institution active state'
        );

        return response()->json([
            'message' => 'Institution status updated.',
            'data' => $institution->fresh(),
        ]);
    }
}
