<?php

namespace App\Http\Controllers;

use App\Models\NationalScore;
use Illuminate\Http\Request;

class NationalScoreController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $q = NationalScore::with('version:id,year,name')
            ->orderByDesc('year')
            ->orderBy('gii_version_id');

        if ($year = $request->query('year')) {
            $q->where('year', (int)$year);
        }
        if ($versionId = $request->query('gii_version_id')) {
            $q->where('gii_version_id', $versionId);
        }
        if ($status = $request->query('status')) {
            $q->where('status', $status);
        }

        $result = $q->paginate($request->query('per_page', 15));

        return response()->json([
            'message' => 'National scores retrieved successfully.',
            'data' => $result,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'gii_version_id' => ['required', 'exists:gii_versions,id'],
            'year' => ['required', 'integer', 'min:1900', 'max:2100'],
            'score' => ['nullable', 'numeric'],
            'global_rank' => ['nullable', 'integer', 'min:1'],
            'regional_rank' => ['nullable', 'integer', 'min:1'],
            'meta' => ['nullable', 'array'],
            'status' => ['nullable', 'in:DRAFT,APPROVED,PUBLISHED'],
        ]);

        $exists = NationalScore::where('gii_version_id', $data['gii_version_id'])
            ->where('year', $data['year'])
            ->exists();

        if ($exists) {
            return response()->json([
                'message' => 'A national score already exists for this version and year.'
            ], 422);
        }

        $nationalScore = NationalScore::create([
            'gii_version_id' => $data['gii_version_id'],
            'year' => $data['year'],
            'score' => $data['score'] ?? null,
            'global_rank' => $data['global_rank'] ?? null,
            'regional_rank' => $data['regional_rank'] ?? null,
            'meta' => $data['meta'] ?? null,
            'status' => $data['status'] ?? 'DRAFT',
        ]);

        return response()->json([
            'message' => 'National score created successfully.',
            'data' => $nationalScore,
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(NationalScore $nationalScore)
    {
        $nationalScore->load('version');

        return response()->json([
            'message' => 'National score retrieved successfully.',
            'data' => $nationalScore,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, NationalScore $nationalScore)
    {
        $data = $request->validate([
            'gii_version_id' => ['sometimes', 'exists:gii_versions,id'],
            'year' => ['sometimes', 'integer', 'min:1900', 'max:2100'],
            'score' => ['nullable', 'numeric'],
            'global_rank' => ['nullable', 'integer', 'min:1'],
            'regional_rank' => ['nullable', 'integer', 'min:1'],
            'meta' => ['nullable', 'array'],
            'status' => ['sometimes', 'in:DRAFT,APPROVED,PUBLISHED'],
        ]);

        $versionId = $data['gii_version_id'] ?? $nationalScore->gii_version_id;
        $year = $data['year'] ?? $nationalScore->year;

        $exists = NationalScore::where('gii_version_id', $versionId)
            ->where('year', $year)
            ->where('id', '!=', $nationalScore->id)
            ->exists();

        if ($exists) {
            return response()->json([
                'message' => 'A national score already exists for this version and year.'
            ], 422);
        }

        $nationalScore->update($data);

        return response()->json([
            'message' => 'National score updated successfully.',
            'data' => $nationalScore->fresh(),
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(NationalScore $nationalScore)
    {
        $nationalScore->delete();

        return response()->json([
            'message' => 'National score deleted successfully.',
        ]);
    }
}
