<?php

namespace App\Http\Controllers;

use App\Models\RawRecord;
use Illuminate\Http\Request;

class RawRecordController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $q = RawRecord::with('submission:id,status,year,month,institution_id')
            ->orderByDesc('created_at');

        if ($submissionId = $request->query('submission_id')) {
            $q->where('submission_id', $submissionId);
        }
        if ($status = $request->query('status')) {
            $q->where('status', $status);
        }
        if ($hash = $request->query('hash')) {
            $q->where('hash', $hash);
        }
        if ($sourceRef = $request->query('source_reference')) {
            $q->where('source_reference', $sourceRef);
        }

        $result = $q->paginate($request->query('per_page', 15));

        return response()->json([
            'message' => 'Raw records retrieved successfully.',
            'data' => $result,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'submission_id' => ['required', 'exists:submissions,id'],
            'payload' => ['required', 'array'],
            'hash' => ['nullable', 'string', 'max:128'],
            'source_reference' => ['nullable', 'string', 'max:255'],
            'status' => ['nullable', 'in:RAW,VALID,INVALID'],
            'notes' => ['nullable', 'string'],
        ]);

        $record = RawRecord::create([
            'submission_id' => $data['submission_id'],
            'payload' => $data['payload'],
            'hash' => $data['hash'] ?? null,
            'source_reference' => $data['source_reference'] ?? null,
            'status' => $data['status'] ?? 'RAW',
            'notes' => $data['notes'] ?? null,
        ]);

        return response()->json([
            'message' => 'Raw record created successfully.',
            'data' => $record,
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(RawRecord $rawRecord)
    {
        $rawRecord->load('submission');

        return response()->json([
            'message' => 'Raw record retrieved successfully.',
            'data' => $rawRecord,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, RawRecord $rawRecord)
    {
        $data = $request->validate([
            'submission_id' => ['sometimes', 'exists:submissions,id'],
            'payload' => ['sometimes', 'array'],
            'hash' => ['nullable', 'string', 'max:128'],
            'source_reference' => ['nullable', 'string', 'max:255'],
            'status' => ['sometimes', 'in:RAW,VALID,INVALID'],
            'notes' => ['nullable', 'string'],
        ]);

        $rawRecord->update($data);

        return response()->json([
            'message' => 'Raw record updated successfully.',
            'data' => $rawRecord->fresh(),
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(RawRecord $rawRecord)
    {
        $rawRecord->delete();

        return response()->json([
            'message' => 'Raw record deleted successfully.',
        ]);
    }
}
