<?php

namespace App\Http\Controllers;

use App\Models\SubmissionFile;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class SubmissionFileController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $q = SubmissionFile::with([
                'submission:id,status,year,month,institution_id',
                'uploader:id,name,email',
            ])
            ->orderByDesc('created_at');

        if ($submissionId = $request->query('submission_id')) {
            $q->where('submission_id', $submissionId);
        }
        if ($fileType = $request->query('file_type')) {
            $q->where('file_type', $fileType);
        }
        if ($uploadedBy = $request->query('uploaded_by')) {
            $q->where('uploaded_by', $uploadedBy);
        }

        $result = $q->paginate($request->query('per_page', 15));

        return response()->json([
            'message' => 'Submission files retrieved successfully.',
            'data' => $result,
        ]);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'submission_id' => ['required', 'exists:submissions,id'],
            'file' => ['required', 'file', 'max:20480'],
            'file_type' => ['nullable', 'in:evidence,dataset,report,annex'],
        ]);

        $file = $request->file('file');
        $path = $file->store("niims/submissions/{$data['submission_id']}", 'public');
        $checksum = hash_file('sha256', $file->getRealPath());

        $saved = SubmissionFile::create([
            'submission_id' => $data['submission_id'],
            'path' => $path,
            'original_name' => $file->getClientOriginalName(),
            'mime' => $file->getMimeType(),
            'size' => $file->getSize(),
            'file_type' => $data['file_type'] ?? 'evidence',
            'checksum' => $checksum,
            'uploaded_by' => $request->user()?->id,
        ]);

        return response()->json([
            'message' => 'Submission file uploaded successfully.',
            'data' => $saved,
        ], 201);
    }

    /**
     * Display the specified resource.
     */
    public function show(SubmissionFile $submissionFile)
    {
        $submissionFile->load(['submission', 'uploader']);

        return response()->json([
            'message' => 'Submission file retrieved successfully.',
            'data' => $submissionFile,
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, SubmissionFile $submissionFile)
    {
        $data = $request->validate([
            'file_type' => ['sometimes', 'in:evidence,dataset,report,annex'],
            'original_name' => ['nullable', 'string', 'max:255'],
            'mime' => ['nullable', 'string', 'max:100'],
            'size' => ['nullable', 'integer', 'min:0'],
            'checksum' => ['nullable', 'string', 'max:64'],
        ]);

        $submissionFile->update($data);

        return response()->json([
            'message' => 'Submission file updated successfully.',
            'data' => $submissionFile->fresh(),
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(SubmissionFile $submissionFile)
    {
        $path = $submissionFile->path;
        $submissionFile->delete();

        if ($path && Storage::disk('public')->exists($path)) {
            Storage::disk('public')->delete($path);
        }

        return response()->json([
            'message' => 'Submission file deleted successfully.',
        ]);
    }
}
