<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('indicator_observations', function (Blueprint $table) {
            $table->id();

            // Which indicator this observation belongs to
            $table->foreignId('gii_indicator_id')
                ->constrained('gii_indicators')
                ->cascadeOnDelete();

            // Which submission produced this value (provenance)
            $table->foreignId('submission_id')
                ->nullable()
                ->constrained('submissions')
                ->nullOnDelete();

            // Time dimension (GII usually yearly; month optional for some indicators)
            $table->unsignedSmallInteger('year')
                ->comment('Reporting year');

            $table->unsignedTinyInteger('month')
                ->nullable()
                ->comment('Optional reporting month (1–12)');

            // The actual measured value
            $table->decimal('value', 18, 6)
                ->nullable()
                ->comment('Standardized indicator value');

            // Unit may be inherited from indicator, but stored here to preserve history
            $table->string('unit', 50)
                ->nullable()
                ->comment('Unit at time of observation');

            // Workflow state for official values
            $table->enum('status', ['DRAFT', 'APPROVED', 'PUBLISHED'])
                ->default('DRAFT')
                ->comment('Observation lifecycle status');

            // Data quality + provenance
            $table->enum('quality_flag', ['OK', 'ESTIMATED', 'IMPUTED', 'OUTLIER', 'MISSING', 'REJECTED'])
                ->default('OK')
                ->comment('Quality classification');

            $table->decimal('confidence', 5, 2)
                ->nullable()
                ->comment('Optional confidence score 0–100');

            $table->string('source', 120)
                ->nullable()
                ->comment('Data source e.g. ZIMSTAT, RBZ, University, SME Portal');

            $table->string('method', 120)
                ->nullable()
                ->comment('Collection method e.g. survey, registry, administrative data');

            $table->json('meta')
                ->nullable()
                ->comment('Extra metadata: raw field mapping, notes, formula refs');

            $table->timestamps();

            // ✅ Uniqueness rule: one observation per indicator per period
            // (month nullable means yearly observations use NULL month — still unique)
            $table->unique(['gii_indicator_id', 'year', 'month'], 'uniq_indicator_period');

            // Helpful indexes for fast reporting + exports
            $table->index(['year', 'status'], 'idx_obs_year_status');
            $table->index(['gii_indicator_id', 'status'], 'idx_obs_indicator_status');
            $table->index('quality_flag', 'idx_obs_quality_flag');
            $table->index('submission_id', 'idx_obs_submission');
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('indicator_observations');
    }
};
